# ----------------------------------------------------------------------------
#
#  Copyright (C) 2014-2021 Fons Adriaensen <fons@linuxaudio.org>
#    
#  This program is free software; you can redistribute it and/or modify
#  it under the terms of the GNU General Public License as published by
#  the Free Software Foundation; either version 3 of the License, or
#  (at your option) any later version.
#
#  This program is distributed in the hope that it will be useful,
#  but WITHOUT ANY WARRANTY; without even the implied warranty of
#  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#  GNU General Public License for more details.
#
#  You should have received a copy of the GNU General Public License
#  along with this program.  If not, see <http:#www.gnu.org/licenses/>.
#
# ----------------------------------------------------------------------------


from zita_audiotools import vresampler_ext


class VResampler() :

    """
    Python bindings to zita-resampler::VResampler
    See the C++ library documentation for details.
    """

    def __init__(self, ratio, nchan, hlen, frel = None):
        """
        Create a new VResampler object.
        """
        if frel is None: frel = 1.0 - 2.6 / hlen
        self._capsule = vresampler_ext.create (self, ratio, nchan, hlen, frel)


    def reset (self):
        """
        Reset the internal state of a VResampler.
        """
        return vresampler_ext.reset (self._capsule)


    def inpsize (self):
        """
        Return inpsize ();

        See the C++ docs for how this can be used.
        """
        return vresampler_ext.inpsize (self._capsule)


    def inpdist (self):
        """
        Return inpdist ();

        See the C++ docs for how this can be used.
        """
        return vresampler_ext.inpdist (self._capsule)


    def set_phase (self, v):
        """
        Set the initial phase of the polyphase filter.
        """
        return vresampler_ext.set_phase (self._capsule, v)
    

    def set_rratio (self, v):
        """
        Set the relative ratio.

        The relative ratio is a small correction to the 
        resampling ratio, used for adaptive resampling.
        """
        return vresampler_ext.set_rratio (self._capsule, v)
    

    def set_rrfilt (self, v):
        """
        Set the relative ratio filter.

        Sets the time constant of the first order filter used
        to apply the relative ratio. 
        """
        return vresampler_ext.set_rrfilt (self._capsule, v)
    

    def process (self, inp, out):
        """
        Resample data.

        The 'inp' and 'out' arguments can each be a 2D-array or
        an int. For a 2D-array, the first dimension is the number
        of frames and the second dimension must be equal to the
        number of channels the VResampler is configured for.
        Using an int for 'inp' or 'out' is equivalent to giving
        a null pointer to the C++ class, with the corresponding
        frame count set to the value of the int. See the C++ docs
        for how this is handled. 

        Returns two ints: 'ninp, nout'. These are equal to the
        number of frames read/written from/to the array arguments.
        Note that this is not the same as what the C++ library
        returns in its argument structure.

        On return either all of the input array has been used, or
        all of the output array has been written. The internal
        state is ready to continue resampling a stream.
        """
        return vresampler_ext.process (self._capsule, inp, out)
    

