"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.default = void 0;
exports.registerOpenSearchRoutes = registerOpenSearchRoutes;
var _lodash = require("lodash");
var _adHelpers = require("./utils/adHelpers");
var _helpers = require("../utils/helpers");
var _opensearchHelpers = require("./utils/opensearchHelpers");
function _defineProperty(e, r, t) { return (r = _toPropertyKey(r)) in e ? Object.defineProperty(e, r, { value: t, enumerable: !0, configurable: !0, writable: !0 }) : e[r] = t, e; }
function _toPropertyKey(t) { var i = _toPrimitive(t, "string"); return "symbol" == typeof i ? i : i + ""; }
function _toPrimitive(t, r) { if ("object" != typeof t || !t) return t; var e = t[Symbol.toPrimitive]; if (void 0 !== e) { var i = e.call(t, r || "default"); if ("object" != typeof i) return i; throw new TypeError("@@toPrimitive must return a primitive value."); } return ("string" === r ? String : Number)(t); } /*
                                                                                                                                                                                                                                                                                                                          * SPDX-License-Identifier: Apache-2.0
                                                                                                                                                                                                                                                                                                                          *
                                                                                                                                                                                                                                                                                                                          * The OpenSearch Contributors require contributions made to
                                                                                                                                                                                                                                                                                                                          * this file be licensed under the Apache-2.0 license or a
                                                                                                                                                                                                                                                                                                                          * compatible open source license.
                                                                                                                                                                                                                                                                                                                          *
                                                                                                                                                                                                                                                                                                                          * Modifications Copyright OpenSearch Contributors. See
                                                                                                                                                                                                                                                                                                                          * GitHub history for details.
                                                                                                                                                                                                                                                                                                                          */
function registerOpenSearchRoutes(apiRouter, opensearchService) {
  apiRouter.get('/_indices', opensearchService.getIndices);
  apiRouter.get('/_indices/{dataSourceId}', opensearchService.getIndices);
  apiRouter.get('/_aliases', opensearchService.getAliases);
  apiRouter.get('/_aliases/{dataSourceId}', opensearchService.getAliases);
  apiRouter.get('/_mappings', opensearchService.getMapping);
  apiRouter.get('/_mappings/{dataSourceId}', opensearchService.getMapping);
  apiRouter.post('/_search', opensearchService.executeSearch);
  apiRouter.put('/create_index', opensearchService.createIndex);
  apiRouter.put('/create_index/{dataSourceId}', opensearchService.createIndex);
  apiRouter.post('/bulk', opensearchService.bulk);
  apiRouter.post('/bulk/{dataSourceId}', opensearchService.bulk);
  apiRouter.post('/delete_index', opensearchService.deleteIndex);
  apiRouter.get('/_remote/info', opensearchService.getClustersInfo);
  apiRouter.get('/_remote/info/', opensearchService.getClustersInfo);
  apiRouter.get('/_remote/info/{dataSourceId}', opensearchService.getClustersInfo);
  apiRouter.get('/_indices_and_aliases', opensearchService.getIndicesAndAliases);
  apiRouter.get('/_indices_and_aliases/{dataSourceId}', opensearchService.getIndicesAndAliases);
}
class OpenSearchService {
  constructor(client, dataSourceEnabled) {
    _defineProperty(this, "client", void 0);
    _defineProperty(this, "dataSourceEnabled", void 0);
    _defineProperty(this, "executeSearch", async (context, request, opensearchDashboardsResponse) => {
      try {
        const {
          index,
          query,
          size = 0,
          sort = undefined,
          collapse = undefined,
          aggs = undefined,
          rawQuery = undefined
        } = request.body;
        const requestBody = rawQuery ? rawQuery : {
          query: query,
          ...(sort !== undefined && {
            sort: sort
          }),
          ...(collapse !== undefined && {
            collapse: collapse
          }),
          ...(aggs !== undefined && {
            aggs: aggs
          })
        };
        const params = {
          index,
          size,
          body: requestBody
        };
        const results = await this.client.asScoped(request).callAsCurrentUser('search', params);
        return opensearchDashboardsResponse.ok({
          body: {
            ok: true,
            response: results
          }
        });
      } catch (err) {
        console.error('Anomaly detector - Unable to execute search', err);
        return opensearchDashboardsResponse.ok({
          body: {
            ok: false,
            error: (0, _adHelpers.getErrorMessage)(err)
          }
        });
      }
    });
    _defineProperty(this, "getIndices", async (context, request, opensearchDashboardsResponse) => {
      const {
        index,
        clusters
      } = request.query;
      const {
        dataSourceId = ''
      } = request.params;
      try {
        const callWithRequest = (0, _helpers.getClientBasedOnDataSource)(context, this.dataSourceEnabled, request, dataSourceId, this.client);
        let indices = [];
        let resolve_resp;
        let response = await callWithRequest('cat.indices', {
          index,
          format: 'json',
          h: 'health,index'
        });
        response = response.map(item => ({
          ...item,
          localCluster: true
        }));

        // only call cat indices
        if (clusters != '') {
          if (index == '') {
            resolve_resp = await callWithRequest('transport.request', {
              method: 'GET',
              path: '/_resolve/index/' + clusters + ':*'
            });
          } else {
            resolve_resp = await callWithRequest('transport.request', {
              method: 'GET',
              path: '/_resolve/index/' + clusters + ':' + index
            });
          }
          indices = resolve_resp.indices.map(item => ({
            index: item.name,
            format: 'json',
            health: 'undefined',
            localCluster: false
          }));
          response = response.concat(indices);
        }
        return opensearchDashboardsResponse.ok({
          body: {
            ok: true,
            response: {
              indices: response
            }
          }
        });
      } catch (err) {
        // In case no matching indices is found it throws an error.
        if (err.statusCode === 404 && (0, _lodash.get)(err, 'body.error.type', '') === 'index_not_found_exception') {
          return opensearchDashboardsResponse.ok({
            body: {
              ok: true,
              response: {
                indices: []
              }
            }
          });
        }
        console.log('Anomaly detector - Unable to get indices', err);
        return opensearchDashboardsResponse.ok({
          body: {
            ok: false,
            error: (0, _adHelpers.getErrorMessage)(err)
          }
        });
      }
    });
    _defineProperty(this, "getAliases", async (context, request, opensearchDashboardsResponse) => {
      const {
        alias
      } = request.query;
      const {
        dataSourceId = ''
      } = request.params;
      try {
        const callWithRequest = (0, _helpers.getClientBasedOnDataSource)(context, this.dataSourceEnabled, request, dataSourceId, this.client);
        const response = await callWithRequest('cat.aliases', {
          alias,
          format: 'json',
          h: 'alias,index'
        });
        return opensearchDashboardsResponse.ok({
          body: {
            ok: true,
            response: {
              aliases: response
            }
          }
        });
      } catch (err) {
        console.log('Anomaly detector - Unable to get aliases', err);
        return opensearchDashboardsResponse.ok({
          body: {
            ok: false,
            error: (0, _adHelpers.getErrorMessage)(err)
          }
        });
      }
    });
    _defineProperty(this, "createIndex", async (context, request, opensearchDashboardsResponse) => {
      const {
        dataSourceId = ''
      } = request.params;

      //@ts-ignore
      const index = request.body.index;
      //@ts-ignore
      const body = request.body.body;
      const callWithRequest = (0, _helpers.getClientBasedOnDataSource)(context, this.dataSourceEnabled, request, dataSourceId, this.client);
      try {
        await callWithRequest('indices.create', {
          index: index,
          body: body
        });
      } catch (err) {
        console.log('Anomaly detector - Unable to create index', err);
        return opensearchDashboardsResponse.ok({
          body: {
            ok: false,
            error: (0, _adHelpers.getErrorMessage)(err)
          }
        });
      }
      try {
        const response = await callWithRequest('cat.indices', {
          index,
          format: 'json',
          h: 'health,index'
        });
        return opensearchDashboardsResponse.ok({
          body: {
            ok: true,
            response: {
              indices: response
            }
          }
        });
      } catch (err) {
        console.log('Anomaly detector - Unable to get indices', err);
        return opensearchDashboardsResponse.ok({
          body: {
            ok: false,
            error: (0, _adHelpers.getErrorMessage)(err)
          }
        });
      }
    });
    _defineProperty(this, "bulk", async (context, request, opensearchDashboardsResponse) => {
      const {
        dataSourceId = ''
      } = request.params;
      const body = request.body;
      try {
        const callWithRequest = (0, _helpers.getClientBasedOnDataSource)(context, this.dataSourceEnabled, request, dataSourceId, this.client);
        const response = await callWithRequest('bulk', {
          body: body
        });
        return opensearchDashboardsResponse.ok({
          body: {
            ok: true,
            response: {
              response
            }
          }
        });
      } catch (err) {
        console.log('Anomaly detector - Unable to perform bulk action', err);
        return opensearchDashboardsResponse.ok({
          body: {
            ok: false,
            error: (0, _adHelpers.getErrorMessage)(err)
          }
        });
      }
    });
    _defineProperty(this, "deleteIndex", async (context, request, opensearchDashboardsResponse) => {
      const index = request.query;
      try {
        await callWithRequest('indices.delete', {
          index: index
        });
      } catch (err) {
        console.log('Anomaly detector - Unable to perform delete index action', err);
        // Ignore the error if it's an index_not_found_exception
        if (!(0, _adHelpers.isIndexNotFoundError)(err)) {
          return opensearchDashboardsResponse.ok({
            body: {
              ok: false,
              error: (0, _adHelpers.getErrorMessage)(err)
            }
          });
        }
      }
      try {
        const response = await this.client.asScoped(request).callAsCurrentUser('cat.indices', {
          index,
          format: 'json',
          h: 'health,index'
        });
        return opensearchDashboardsResponse.ok({
          body: {
            ok: true,
            response: {
              indices: response
            }
          }
        });
      } catch (err) {
        console.log('Anomaly detector - Unable to get indices', err);
        return opensearchDashboardsResponse.ok({
          body: {
            ok: false,
            error: (0, _adHelpers.getErrorMessage)(err)
          }
        });
      }
    });
    _defineProperty(this, "getMapping", async (context, request, opensearchDashboardsResponse) => {
      let {
        indices
      } = request.query;
      // If indices is not an array, convert it to an array, server framework auto converts single item in string array to a string
      if (!Array.isArray(indices)) {
        indices = [indices];
      }
      const {
        dataSourceId = ''
      } = request.params;
      try {
        const callWithRequest = (0, _helpers.getClientBasedOnDataSource)(context, this.dataSourceEnabled, request, dataSourceId, this.client);
        let mappings = {};
        let remoteMappings = {};
        let localIndices = indices.filter(index => !index.includes(':'));
        let remoteIndices = indices.filter(index => index.includes(':'));
        if (localIndices.length > 0) {
          mappings = await callWithRequest('indices.getMapping', {
            index: localIndices
          });
        }

        // make call to fields_caps
        if (remoteIndices.length) {
          const fieldCapsResponse = await callWithRequest('fieldCaps', {
            index: remoteIndices.toString(),
            fields: '*',
            include_unmapped: true
          });
          remoteMappings = (0, _opensearchHelpers.convertFieldCapsToMappingStructure)(fieldCapsResponse);
        }
        Object.assign(mappings, remoteMappings);
        return opensearchDashboardsResponse.ok({
          body: {
            ok: true,
            response: {
              mappings: mappings
            }
          }
        });
      } catch (err) {
        console.log('Anomaly detector - Unable to get mappings', err);
        return opensearchDashboardsResponse.ok({
          body: {
            ok: false,
            error: (0, _adHelpers.getErrorMessage)(err)
          }
        });
      }
    });
    // we use this to retrieve indices and aliases from both the local cluster and remote clusters
    // 3 different OS APIs are called here, _cat/indices, _cat/aliases and _resolve/index
    _defineProperty(this, "getIndicesAndAliases", async (context, request, opensearchDashboardsResponse) => {
      const {
        indexOrAliasQuery,
        clusters,
        queryForLocalCluster
      } = request.query;
      const {
        dataSourceId = ''
      } = request.params;
      try {
        const callWithRequest = (0, _helpers.getClientBasedOnDataSource)(context, this.dataSourceEnabled, request, dataSourceId, this.client);
        let indicesResponse = [];
        let aliasesResponse = [];
        if (queryForLocalCluster == 'true') {
          indicesResponse = await callWithRequest('cat.indices', {
            index: indexOrAliasQuery,
            format: 'json',
            h: 'health,index'
          });
          indicesResponse = indicesResponse.map(item => ({
            ...item,
            localCluster: true
          }));
          aliasesResponse = await callWithRequest('cat.aliases', {
            alias: indexOrAliasQuery,
            format: 'json',
            h: 'alias,index'
          });
          aliasesResponse = aliasesResponse.map(item => ({
            ...item,
            localCluster: true
          }));
        }

        // only call cat indices and cat aliases
        if (clusters != '') {
          let remoteIndices = [];
          let remoteAliases = [];
          let resolveResponse;
          const resolveIndexQuery = indexOrAliasQuery == '' ? clusters.split(',').map(cluster => `${cluster}:*`).join(',') : clusters.split(',').map(cluster => `${cluster}:${indexOrAliasQuery}`).join(',');
          resolveResponse = await callWithRequest('transport.request', {
            method: 'GET',
            path: '/_resolve/index/' + resolveIndexQuery
          });
          remoteIndices = resolveResponse.indices.map(item => ({
            index: item.name,
            format: 'json',
            health: 'undefined',
            localCluster: false
          }));
          remoteAliases = resolveResponse.aliases.map(item => ({
            alias: item.name,
            index: item.indices,
            format: 'json',
            localCluster: false
          }));
          indicesResponse = indicesResponse.concat(remoteIndices);
          aliasesResponse = aliasesResponse.concat(remoteAliases);
        }
        return opensearchDashboardsResponse.ok({
          body: {
            ok: true,
            response: {
              aliases: aliasesResponse,
              indices: indicesResponse
            }
          }
        });
      } catch (err) {
        // In case no matching indices is found it throws an error.
        if (err.statusCode === 404 && (0, _lodash.get)(err, 'body.error.type', '') === 'index_not_found_exception') {
          return opensearchDashboardsResponse.ok({
            body: {
              ok: true,
              response: {
                indices: [],
                aliases: []
              }
            }
          });
        }
        console.log('Anomaly detector - Unable to get indices and aliases', err);
        return opensearchDashboardsResponse.ok({
          body: {
            ok: false,
            error: (0, _adHelpers.getErrorMessage)(err)
          }
        });
      }
    });
    _defineProperty(this, "getClustersInfo", async (context, request, opensearchDashboardsResponse) => {
      const {
        dataSourceId = ''
      } = request.params;
      try {
        const callWithRequest = (0, _helpers.getClientBasedOnDataSource)(context, this.dataSourceEnabled, request, dataSourceId, this.client);
        let clustersResponse = [];
        try {
          const remoteInfo = await callWithRequest('transport.request', {
            method: 'GET',
            path: '/_remote/info'
          });
          clustersResponse = Object.keys(remoteInfo).map(key => ({
            name: key,
            localCluster: false
          }));
        } catch (remoteErr) {
          console.warn('Failed to fetch remote cluster info, proceeding with local datasource info only.', remoteErr);
        }
        const clusterHealth = await callWithRequest('cat.health', {
          format: 'json',
          h: 'cluster'
        });
        clustersResponse.push({
          name: clusterHealth[0].cluster,
          localCluster: true
        });
        return opensearchDashboardsResponse.ok({
          body: {
            ok: true,
            response: {
              clusters: clustersResponse
            }
          }
        });
      } catch (err) {
        console.error('Alerting - OpensearchService - getClusterHealth:', err);
        return opensearchDashboardsResponse.ok({
          body: {
            ok: false,
            error: (0, _adHelpers.getErrorMessage)(err)
          }
        });
      }
    });
    this.client = client;
    this.dataSourceEnabled = dataSourceEnabled;
  }
}
exports.default = OpenSearchService;
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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