"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.SecuritySavedObjectsClientWrapper = void 0;

var _lodash = _interopRequireDefault(require("lodash"));

var _common = require("../../common");

function _interopRequireDefault(obj) { return obj && obj.__esModule ? obj : { default: obj }; }

function _defineProperty(obj, key, value) { if (key in obj) { Object.defineProperty(obj, key, { value: value, enumerable: true, configurable: true, writable: true }); } else { obj[key] = value; } return obj; }

class SecuritySavedObjectsClientWrapper {
  constructor() {
    _defineProperty(this, "httpStart", void 0);

    _defineProperty(this, "config", void 0);

    _defineProperty(this, "wrapperFactory", wrapperOptions => {
      var _state$authInfo;

      const state = this.httpStart.auth.get(wrapperOptions.request).state || {};
      const selectedTenant = state.selectedTenant;
      const username = (_state$authInfo = state.authInfo) === null || _state$authInfo === void 0 ? void 0 : _state$authInfo.user_name;
      const isGlobalEnabled = this.config.multitenancy.tenants.enable_global;
      const isPrivateEnabled = this.config.multitenancy.tenants.enable_private;
      let namespaceValue = selectedTenant;

      const createWithNamespace = async (type, attributes, options) => {
        namespaceValue = this.getNamespaceValue(selectedTenant, isPrivateEnabled, username);

        _lodash.default.assign(options, {
          namespace: [namespaceValue]
        });

        return await wrapperOptions.client.create(type, attributes, options);
      };

      const bulkGetWithNamespace = async (objects = [], options = {}) => {
        namespaceValue = this.getNamespaceValue(selectedTenant, isPrivateEnabled, username);

        _lodash.default.assign(options, {
          namespace: [namespaceValue]
        });

        return await wrapperOptions.client.bulkGet(objects, options);
      };

      const findWithNamespace = async options => {
        var _state$authInfo2;

        const tenants = (_state$authInfo2 = state.authInfo) === null || _state$authInfo2 === void 0 ? void 0 : _state$authInfo2.tenants;
        const availableTenantNames = Object.keys(tenants);
        availableTenantNames.push(_common.DEFAULT_TENANT); // The value of namespace is "default" if saved objects are created when opensearch_security.multitenancy.enable_aggregation_view is set to false. So adding it to find.

        if (isGlobalEnabled) {
          availableTenantNames.push(_common.GLOBAL_TENANT_SYMBOL);
        }

        if (isPrivateEnabled) {
          availableTenantNames.push(_common.PRIVATE_TENANT_SYMBOL + username);
        }

        if (availableTenantNames.includes(_common.globalTenantName)) {
          let index = availableTenantNames.indexOf(_common.globalTenantName);

          if (index > -1) {
            availableTenantNames.splice(index, 1);
          }

          index = availableTenantNames.indexOf(username);

          if (index > -1) {
            availableTenantNames.splice(index, 1);
          }
        }

        if ((0, _common.isPrivateTenant)(selectedTenant)) {
          namespaceValue = selectedTenant + username;
        }

        if (!!options.namespaces) {
          const namespacesToInclude = Array.isArray(options.namespaces) ? options.namespaces : [options.namespaces];
          const typeToNamespacesMap = {};
          const searchTypes = Array.isArray(options.type) ? options.type : [options.type];
          searchTypes.forEach(t => {
            typeToNamespacesMap[t] = namespacesToInclude;
          });

          if (searchTypes.includes('config')) {
            if (namespacesToInclude.includes(namespaceValue)) {
              typeToNamespacesMap.config = [namespaceValue];
            } else {
              delete typeToNamespacesMap.config;
            }
          }

          options.typeToNamespacesMap = new Map(Object.entries(typeToNamespacesMap));
          options.type = '';
          options.namespaces = [];
        } else {
          options.namespaces = [namespaceValue];
        }

        return await wrapperOptions.client.find(options);
      };

      const getWithNamespace = async (type, id, options = {}) => {
        namespaceValue = this.getNamespaceValue(selectedTenant, isPrivateEnabled, username);

        _lodash.default.assign(options, {
          namespace: [namespaceValue]
        });

        return await wrapperOptions.client.get(type, id, options);
      };

      const updateWithNamespace = async (type, id, attributes, options = {}) => {
        namespaceValue = this.getNamespaceValue(selectedTenant, isPrivateEnabled, username);

        _lodash.default.assign(options, {
          namespace: [namespaceValue]
        });

        return await wrapperOptions.client.update(type, id, attributes, options);
      };

      const bulkCreateWithNamespace = async (objects, options) => {
        namespaceValue = this.getNamespaceValue(selectedTenant, isPrivateEnabled, username);

        _lodash.default.assign(options, {
          namespace: [namespaceValue]
        });

        return await wrapperOptions.client.bulkCreate(objects, options);
      };

      const bulkUpdateWithNamespace = async (objects, options) => {
        namespaceValue = this.getNamespaceValue(selectedTenant, isPrivateEnabled, username);

        _lodash.default.assign(options, {
          namespace: [namespaceValue]
        });

        return await wrapperOptions.client.bulkUpdate(objects, options);
      };

      const deleteWithNamespace = async (type, id, options = {}) => {
        namespaceValue = this.getNamespaceValue(selectedTenant, isPrivateEnabled, username);

        _lodash.default.assign(options, {
          namespace: [namespaceValue]
        });

        return await wrapperOptions.client.delete(type, id, options);
      };

      const checkConflictsWithNamespace = async (objects = [], options = {}) => {
        namespaceValue = this.getNamespaceValue(selectedTenant, isPrivateEnabled, username);

        _lodash.default.assign(options, {
          namespace: [namespaceValue]
        });

        return await wrapperOptions.client.checkConflicts(objects, options);
      };

      return { ...wrapperOptions.client,
        get: getWithNamespace,
        update: updateWithNamespace,
        bulkCreate: bulkCreateWithNamespace,
        bulkGet: bulkGetWithNamespace,
        bulkUpdate: bulkUpdateWithNamespace,
        create: createWithNamespace,
        delete: deleteWithNamespace,
        errors: wrapperOptions.client.errors,
        checkConflicts: checkConflictsWithNamespace,
        addToNamespaces: wrapperOptions.client.addToNamespaces,
        find: findWithNamespace,
        deleteFromNamespaces: wrapperOptions.client.deleteFromNamespaces
      };
    });
  }

  isAPrivateTenant(selectedTenant, isPrivateEnabled) {
    return selectedTenant !== undefined && isPrivateEnabled && (0, _common.isPrivateTenant)(selectedTenant);
  }

  getNamespaceValue(selectedTenant, isPrivateEnabled, username) {
    let namespaceValue = selectedTenant;

    if (this.isAPrivateTenant(selectedTenant, isPrivateEnabled)) {
      namespaceValue = selectedTenant + username;
    }

    return namespaceValue;
  }

}

exports.SecuritySavedObjectsClientWrapper = SecuritySavedObjectsClientWrapper;
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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