"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const chai_1 = require("chai");
const path = require("path");
const getAnsibleMetaData_1 = require("../../src/utils/getAnsibleMetaData");
const helper_1 = require("../helper");
function getAnsibleTestInfo() {
    const ansibleInfo = {};
    ansibleInfo["core version"] = ".";
    ansibleInfo["location"] = "/ansible";
    (ansibleInfo["config file path"] = path.resolve(__dirname, "..", "fixtures", "utils", "getAnsibleMetaData", "ansible.cfg")),
        (ansibleInfo["collections location"] = [
            path.resolve(__dirname, "..", "fixtures", "common", "collections"),
        ]);
    ansibleInfo["module location"] = ["/modules"];
    ansibleInfo["default host list path"] = [
        path.resolve(__dirname, "..", "fixtures", "utils", "getAnsibleMetaData", "inventory"),
    ];
    return ansibleInfo;
}
function getPythonTestInfo() {
    const pythonInfo = {};
    pythonInfo["version"] = ".";
    pythonInfo["location"] = "/python";
    return pythonInfo;
}
function getAnsibleLintTestInfo() {
    const ansibleLintInfo = {};
    ansibleLintInfo["version"] = ".";
    ansibleLintInfo["upgrade status"] = "A new version"; // this key will be undefined (but the key will be present) because the value only gets updated based on the ansible-lint version used
    ansibleLintInfo["location"] = "/ansible-lint";
    ansibleLintInfo["config file path"] = "/.ansible-lint"; // this key will be undefined (but the key will be present) because the value only gets updated when validation in run
    return ansibleLintInfo;
}
function getExecutionEnvironmentTestInfo() {
    const eeInfo = {};
    eeInfo["container engine"] = ["docker", "podman"];
    eeInfo["container image"] = "ghcr.io/";
    eeInfo["container volume mounts"] = [
        {
            src: "/fixtures/common/collections",
            dest: "/fixtures/common/collections",
        },
    ];
    return eeInfo;
}
function testCommands() {
    describe("Verify the working of command executions", () => {
        const tests = [
            {
                args: ["ansible", "--version"],
                result: "configured module search path",
            },
            {
                args: ["python3", "--version"],
                result: "Python",
            },
            {
                args: ["ansible-lint", "--version"],
                result: "using ansible",
            },
            {
                args: ["ansible-playbook", "missing-file"],
                result: undefined,
            },
        ];
        tests.forEach(({ args, result }) => {
            it(`should return result for '${args.join(" ")}'`, async function () {
                const output = await (0, getAnsibleMetaData_1.getResultsThroughCommandRunner)(args[0], args[1]);
                if (result === undefined) {
                    (0, chai_1.expect)(output).to.be.undefined;
                }
                else {
                    (0, chai_1.expect)(output?.stdout).contains(result);
                }
            });
        });
    });
}
describe("getAnsibleMetaData()", () => {
    const workspaceManager = (0, helper_1.createTestWorkspaceManager)();
    const fixtureFilePath = "utils/getAnsibleMetaData/plays.yml";
    const fixtureFileUri = (0, helper_1.resolveDocUri)(fixtureFilePath);
    const context = workspaceManager.getContext(fixtureFileUri);
    const textDoc = (0, helper_1.getDoc)(fixtureFilePath);
    const docSettings = context?.documentSettings.get(textDoc.uri);
    let actualAnsibleMetaData = {};
    let ansibleInfoForTest = {};
    let pythonInfoForTest = {};
    let ansibleLintInfoForTest = {};
    let executionEnvironmentInfoForTest = {};
    describe("With EE disabled", () => {
        before(async () => {
            if (context !== undefined) {
                actualAnsibleMetaData = await (0, getAnsibleMetaData_1.getAnsibleMetaData)(context, undefined);
            }
            ansibleInfoForTest = getAnsibleTestInfo();
            pythonInfoForTest = getPythonTestInfo();
            ansibleLintInfoForTest = getAnsibleLintTestInfo();
        });
        describe("Verify ansible details", () => {
            it("should contain all the keys for ansible information", function () {
                if (actualAnsibleMetaData["ansible information"]) {
                    (0, chai_1.expect)(Object.keys(ansibleInfoForTest).length).equals(Object.keys(actualAnsibleMetaData["ansible information"]).length);
                }
                else {
                    (0, chai_1.expect)(false);
                }
            });
            it("should have information about ansible version used", function () {
                if (actualAnsibleMetaData["ansible information"]) {
                    (0, chai_1.expect)(actualAnsibleMetaData["ansible information"]["core version"]).includes(ansibleInfoForTest["core version"]);
                }
                else {
                    (0, chai_1.expect)(false);
                }
            });
            it("should have a valid ansible location", function () {
                if (actualAnsibleMetaData["ansible information"]) {
                    (0, chai_1.expect)(actualAnsibleMetaData["ansible information"]["location"]).include(ansibleInfoForTest["location"]);
                }
                else {
                    (0, chai_1.expect)(false);
                }
            });
            it("should have a valid config file location", function () {
                if (actualAnsibleMetaData["ansible information"]) {
                    (0, chai_1.expect)(actualAnsibleMetaData["ansible information"]["config file path"]).to.include(ansibleInfoForTest["config file path"]);
                }
                else {
                    (0, chai_1.expect)(false);
                }
            });
            it("should have a valid collections location", function () {
                const x = ansibleInfoForTest["collections location"];
                if (Array.isArray(x)) {
                    if (actualAnsibleMetaData["ansible information"]) {
                        (0, chai_1.expect)(actualAnsibleMetaData["ansible information"]["collections location"]).to.include.members(x);
                    }
                    else {
                        (0, chai_1.expect)(false);
                    }
                }
            });
            it("should have a valid inventory file path", function () {
                const x = ansibleInfoForTest["default host list path"];
                if (Array.isArray(x) && actualAnsibleMetaData["ansible information"]) {
                    (0, chai_1.expect)(actualAnsibleMetaData["ansible information"]["default host list path"]).to.include.members(x);
                }
                else {
                    (0, chai_1.expect)(false);
                }
            });
        });
        describe("Verify python details", () => {
            it("should contain all the keys for python information", function () {
                if (actualAnsibleMetaData["python information"]) {
                    (0, chai_1.expect)(Object.keys(pythonInfoForTest).length).equals(Object.keys(actualAnsibleMetaData["python information"]).length);
                }
                else {
                    (0, chai_1.expect)(false);
                }
            });
            it("should have information about python version used", function () {
                if (actualAnsibleMetaData["python information"]) {
                    (0, chai_1.expect)(actualAnsibleMetaData["python information"]["version"]).includes(pythonInfoForTest["version"]);
                }
                else {
                    (0, chai_1.expect)(false);
                }
            });
            it("should have a valid python location", function () {
                if (actualAnsibleMetaData["python information"]) {
                    (0, chai_1.expect)(actualAnsibleMetaData["python information"]["location"]).include(pythonInfoForTest["location"]);
                }
                else {
                    (0, chai_1.expect)(false);
                }
            });
        });
        describe("Verify ansible-lint details", () => {
            it("should contain all the keys for ansible-lint information", function () {
                if (actualAnsibleMetaData["ansible-lint information"]) {
                    (0, chai_1.expect)(Object.keys(ansibleLintInfoForTest).length).equals(Object.keys(actualAnsibleMetaData["ansible-lint information"])
                        .length);
                }
                else {
                    (0, chai_1.expect)(false);
                }
            });
            it("should have information about ansible-lint version used", function () {
                if (actualAnsibleMetaData["ansible-lint information"]) {
                    (0, chai_1.expect)(actualAnsibleMetaData["ansible-lint information"]["version"]).includes(ansibleLintInfoForTest["version"]);
                }
                else {
                    (0, chai_1.expect)(false);
                }
            });
            it("should have a valid ansible-lint location", function () {
                if (actualAnsibleMetaData["ansible-lint information"]) {
                    (0, chai_1.expect)(actualAnsibleMetaData["ansible-lint information"]["location"]).include(ansibleLintInfoForTest["location"]);
                }
                else {
                    (0, chai_1.expect)(false);
                }
            });
        });
        describe("Verify the absence of execution environment details", () => {
            it("should not contain execution environment details", function () {
                (0, chai_1.expect)(actualAnsibleMetaData["execution environment information"]).to.be
                    .undefined;
            });
        });
        testCommands();
    });
    describe("With EE enabled @ee", () => {
        before(async () => {
            if (docSettings) {
                await (0, helper_1.enableExecutionEnvironmentSettings)(docSettings);
            }
            if (context) {
                actualAnsibleMetaData = await (0, getAnsibleMetaData_1.getAnsibleMetaData)(context, undefined);
            }
            ansibleInfoForTest = getAnsibleTestInfo();
            pythonInfoForTest = getPythonTestInfo();
            ansibleLintInfoForTest = getAnsibleLintTestInfo();
            executionEnvironmentInfoForTest = getExecutionEnvironmentTestInfo();
        });
        describe("Verify the presence of execution environment details", () => {
            it("should have a valid container engine", function () {
                if (actualAnsibleMetaData["execution environment information"]) {
                    (0, chai_1.expect)(executionEnvironmentInfoForTest["container engine"]).to.include(actualAnsibleMetaData["execution environment information"]["container engine"]);
                }
                else {
                    (0, chai_1.expect)(false);
                }
            });
            it("should have a valid container image", function () {
                if (actualAnsibleMetaData["execution environment information"]) {
                    (0, chai_1.expect)(actualAnsibleMetaData["execution environment information"]["container image"]).to.include(executionEnvironmentInfoForTest["container image"]);
                }
                else {
                    (0, chai_1.expect)(false);
                }
            });
            after(async () => {
                if (docSettings) {
                    await (0, helper_1.disableExecutionEnvironmentSettings)(docSettings);
                }
            });
        });
        testCommands();
    });
});
