<!--
title: Serverless Framework - AWS Lambda Events - S3
menuText: S3
menuOrder: 5
description:  Setting up AWS S3 Events with AWS Lambda via the Serverless Framework
layout: Doc
-->

<!-- DOCS-SITE-LINK:START automatically generated  -->

### [Read this on the main serverless docs site](https://www.serverless.com/framework/docs/providers/aws/events/s3)

<!-- DOCS-SITE-LINK:END -->

# S3

## Simple event definition

This will create a `photos` bucket which fires the `resize` function when an object is added or modified inside the bucket. A hardcoded bucket name can lead to issues as a bucket name can only be used once in S3. For that you can use the [Serverless Variable syntax](../guide/variables.md) and add dynamic elements to the bucket name.

```yaml
functions:
  resize:
    handler: resize.handler
    events:
      - s3: photos
```

## Setting the specific trigger event

This will create a bucket `photos`. The `users` function is called whenever an object is removed from the bucket. Check out the [AWS documentation](http://docs.aws.amazon.com/AmazonS3/latest/dev/NotificationHowTo.html#notification-how-to-event-types-and-destinations) to learn more about all the different event types that can be configured.

```yaml
functions:
  users:
    handler: users.handler
    events:
      - s3:
          bucket: photos
          event: s3:ObjectRemoved:*
```

## Setting filter rules

This will create a bucket `photos`. The `users` function is called whenever an image with `.jpg` extension is uploaded to folder `uploads` in the bucket. Check out the [AWS documentation](http://docs.aws.amazon.com/AmazonS3/latest/dev/NotificationHowTo.html#notification-how-to-filtering) to learn more about all the different filter types that can be configured.

```yaml
functions:
  users:
    handler: users.handler
    events:
      - s3:
          bucket: photos
          event: s3:ObjectCreated:*
          rules:
            - prefix: uploads/
            - suffix: .jpg
```

## Triggering separate functions from the same bucket

You're able to repeat the S3 event configuration in the same or separate functions so one bucket can call these functions. One caveat though is that you can't repeat the same configuration in both functions, e.g. the event type has to be different.

The following example will work:

```yaml
functions:
  users:
    handler: users.handler
    events:
      - s3:
          bucket: photos
          event: s3:ObjectCreated:*
      - s3:
          bucket: photos
          event: s3:ObjectRemoved:*
```

## Custom bucket configuration

If you need to configure the bucket itself, you'll need to add s3 resources to the provider configuration:

```yaml
functions:
  resize:
    handler: resize.handler
    events:
      - s3: bucketOne

provider:
  s3:
    bucketOne:
      name: my-custom-bucket-name
      # Eventual additional properties in camel case
```

All additional bucket properties could be found in the Cloud Formation documentation: https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/aws-properties-s3-bucket.html

**NOTE:** The properties could be camelCase and not PascalCase, i.e: use `bucketEncryption` over `BucketEncryption`. Sub properties should be according to the specification above:

```yml
provider:
  s3:
    bucketOne:
      name: my-custom-bucket-name
      versioningConfiguration:
        Status: Enabled
```

**NOTE:** If you use an `provider.s3` object and don't specify a `name` or `bucketName` property, the key will be used as a bucket name. The key must follow the naming conventions of s3 buckets, please see "Rules for Bucket Naming": https://docs.aws.amazon.com/AmazonS3/latest/dev/BucketRestrictions.html

## Using existing buckets

Sometimes you might want to attach Lambda functions to existing S3 buckets. In that case you just need to set the `existing` event configuration property to `true`. All the other config parameters can also be used on existing buckets:

**IMPORTANT:** You can only attach 1 existing S3 bucket per function.

**NOTE:** Using the `existing` config will add an additional Lambda function and IAM Role to your stack. The Lambda function backs-up the Custom S3 Resource which is used to support existing S3 buckets.

**NOTE:** This property was added in version 1.47.0. Older versions don't support this feature.

```yaml
functions:
  users:
    handler: users.handler
    events:
      - s3:
          bucket: legacy-photos
          event: s3:ObjectCreated:*
          rules:
            - prefix: uploads/
            - suffix: .jpg
          existing: true
```

## Forcing deploying of triggers

An S3 bucket with triggers attached may not be correctly updated by AWS Cloudformation on subsequent deployments. To circumvent this issue you can use the `forceDeploy` flag which will try to force Cloudformation to update the triggers no matter what. This has to be used in conjuction with the `existing: true` flag.

```yml
functions:
  users:
    handler: users.handler
    events:
      - s3:
          bucket: legacy-photos
          event: s3:ObjectCreated:*
          rules:
            - prefix: uploads/
            - suffix: .jpg
          existing: true
          forceDeploy: true
```
